rm(list=ls())
gc()

setwd("")

library(sna) ; library(openxlsx) ; library(arules) 

# ---- data: organizations by concepts ----

# reading in bipartite actorsXconcepts networks (subtract)

bipnet1 <- as.matrix(read.xlsx("bipnet_subtr_010115-170515_1.xlsx", rowNames = T))
bipnet2 <- as.matrix(read.xlsx("bipnet_subtr_180515-300815_2.xlsx", rowNames = T))
bipnet3 <- as.matrix(read.xlsx("bipnet_subtr_310815-191015_3.xlsx", rowNames = T))

colnames(bipnet1) <- gsub("\\.", " ", colnames(bipnet1))
colnames(bipnet2) <- gsub("\\.", " ", colnames(bipnet2))
colnames(bipnet3) <- gsub("\\.", " ", colnames(bipnet3))

# ---- block model (Jaccard similarity) ----

# reading in data.frame for actor types

actors <- read.xlsx("actors.xlsx", colNames = T)
head(actors)

at <- car::recode(actors$type, 
                  recodes = "'incm' = '1'; 'dm' = '2'; 'engo' = '3'; 'research' = '4'; 
                  'reg_mcp' = '4'; 'pol_1' = '4'; 'pol_2' = '4'; 'ngo' = '4'")

data.frame(actors$org, at)

stage <- c("(Stage 1)", "(Stage 2)", "(Stage 3)")
group.names <- c("incumbents", "GPs", "ENGOs", "RG")
bipnet.list <- list(bipnet1, bipnet2, bipnet3)

for (k in 1:length(bipnet.list)) {
  
  # bipartite actorsXconcepts network for particular stage
  m <- as.matrix(bipnet.list[[k]]) 
  
  # dichotomization of the bipartite actorsXconcepts network
  m <- ifelse(m > 0, 1, 0)
 
  # matching the present actors with the actor types
  at.i <- match(rownames(m), actors$org)
  # checking whether everything matches
  rownames(m)[which(is.na(match(rownames(m), actors$org))==T)]
  # saving the matches into the predefined actor groups
  groups <- at[at.i] 
  
  # calculating the distance matrix based on Jaccard coefficient
  dm <- as.matrix(arules::dissimilarity(m, method = "Jaccard"))
  
  # getting proximity matrix for easier interpretation
  dm <- 1 - dm
  
  # block model
  
  # the proximity matrix is blocked based on the four predefined actor groups
  # the cell values are within- and between-block means of the Jaccard similarity coefficient 
  
  bm <- blockmodel(dm, ec = groups, mode = "graph")
  bm
  
  # resulting block model
  round(bm$block.model, 3)
  
  # simulating a sampling distribution consisting of Jaccard similarity values obtained from 5000 random bipartite networks 
  # random networks have same number of nodes and are conditioned on density of the observed network
  
  jacc.dist <- c()
  set.seed(666)
  for (i in 1:5000) {
    
    # random bipartite graph
    rg <- igraph::sample_bipartite(n1 = nrow(m), n2 = ncol(m), p=sum(m)/(nrow(m)*ncol(m)))
    im <- igraph::as_incidence_matrix(rg)
    # Jaccard similarity coefficient matrix
    output <- as.matrix(arules::dissimilarity(im, method = "Jaccard"))
    jacc.mat <- 1 - output 
    jacc.dist[i] <- mean(jacc.mat) # overall (network) average
    
  }
  
  # constructing confidence intervals as 0.025 and 0.975 quantiles of the simulated sampling distribution
  par(mfrow=c(1,1))
  ci.l <- quantile(jacc.dist, probs = 0.025) # confidence interval lower bound
  ci.u <- quantile(jacc.dist, probs = 0.975) # confidence interval upper bound
  hist(jacc.dist, xlim = c(0,1))             # plotting the simulated sampling distribution
  abline(v=ci.l, lwd=2)                      # confidence interval lower bound
  abline(v=ci.u, lwd=2)                      # confidence interval upper bound
  
  # visualization of image graph
  
  # saving image matrix
  bm.mat <- bm$block.model 
  
  # fixing coordinates of the image matrix/graph
  coord <- c(0,2,0,1,1,2,1,1)
  coord.mat <- matrix(coord, nrow=4, byrow=T)
  
  # color coding of the image matrix based on the statistical in/significance (sig.) of the relationships
  bme.col <- matrix(rep("", nrow(bm.mat)*ncol(bm.mat)), nrow=nrow(bm.mat))
  bme.col[bm.mat>=ci.u] <- "lightgreen"               # sig. higher than the overall average
  bme.col[bm.mat<=ci.l] <- "red"                      # sig. lower than the overall average
  bme.col[bm.mat>ci.l & bm.mat<ci.u] <- "lightgrey"   # not sig. different than the overall average
  bme.col
  
  # plotting the block image graph
  gplot(matrix(rep(1, 16), 4, 4), 
        usearrows = F, 
        label = group.names, 
        label.cex = 2,
        vertex.col = c("orangered", "lightblue", "lightgreen", "pink"),
        vertex.cex = 2, 
        label.pos = 5, 
        vertex.border = F,
        edge.col = bme.col, 
        edge.lwd = 5, 
        diag = T,
        coord = coord.mat, 
        jitter = F,
        main = paste0("Block matrix image: Jaccard similarity ", stage[k])
        )
  
}

# ---- block model: Dice similarity ----

# using Dice similarity coefficient instead of Jaccard coefficient

for (k in 1:length(bipnet.list)) {
  
  # bipartite actorsXconcepts network for particular stage
  m <- as.matrix(bipnet.list[[k]]) 
  
  # dichotomization of the bipartite actorsXconcepts network
  m <- ifelse(m > 0, 1, 0)
  
  # matching present actors with the actor types
  at.i <- match(rownames(m), actors$org)
  # checking whether everything matches
  rownames(m)[which(is.na(match(rownames(m), actors$org))==T)]
  # saving the matches into the predefined actor groups
  groups <- at[at.i] 
  
  # calculating the distance matrix based on Dice coeffcient
  dm <- as.matrix(arules::dissimilarity(m, method = "dice"))
  
  # getting proximity matrix for easier interpretation
  dm <- 1 - dm
  
  # block model
  
  # the proximity matrix is blocked based on the four predefined actor groups
  # the cell values are within- and between-block means of the Dice similarity coefficient 
  
  bm <- blockmodel(dm, ec = groups, mode = "graph")
  bm
  
  # resulting block model
  round(bm$block.model, 3)
  
  # simulating a sampling distribution consisting of Dice similarity values obtained from 5000 random bipartite networks 
  # random networks have same number of nodes and are conditioned on density of the observed network
  
  dice.dist <- c()
  set.seed(666)
  for (i in 1:5000) {
    
    # random bipartite graph
    rg <- igraph::sample_bipartite(n1 = nrow(m), n2 = ncol(m), p=sum(m)/(nrow(m)*ncol(m)))
    im <- igraph::as_incidence_matrix(rg)
    # Jaccard similarity coefficient matrix
    output <- as.matrix(arules::dissimilarity(im, method = "dice"))
    dice.mat <- 1 - output 
    dice.dist[i] <- mean(dice.mat) # overall (network) average
    
  }
  
  # constructing confidence intervals as 0.025 and 0.975 quantiles of the simulated sampling distribution
  par(mfrow=c(1,1))
  ci.l <- quantile(dice.dist, probs = 0.025) # confidence interval lower bound
  ci.u <- quantile(dice.dist, probs = 0.975) # confidence interval upper bound
  hist(dice.dist, xlim = c(0,1))             # plotting the simulated sampling distribution
  abline(v=ci.l, lwd=2)                      # confidence interval lower bound
  abline(v=ci.u, lwd=2)                      # confidence interval upper bound
  
  # visualization of image graph
  
  # saving image matrix
  bm.mat <- bm$block.model 
  
  # fixing coordinates of the image matrix/graph
  coord <- c(0,2,0,1,1,2,1,1)
  coord.mat <- matrix(coord, nrow=4, byrow=T)
  
  # color coding of the image matrix based on the statistical in/significance (sig.) of the relationships
  bme.col <- matrix(rep("", nrow(bm.mat)*ncol(bm.mat)), nrow=nrow(bm.mat))
  bme.col[bm.mat>=ci.u] <- "lightgreen"               # sig. higher than the overall average
  bme.col[bm.mat<=ci.l] <- "red"                      # sig. lower than the overall average
  bme.col[bm.mat>ci.l & bm.mat<ci.u] <- "lightgrey"   # not sig. different than the overall average
  bme.col
  
  # plotting the image graph
  gplot(matrix(rep(1, 16), 4, 4), 
        usearrows = F, 
        label = group.names, 
        label.cex = 2,
        vertex.col = c("orangered", "lightblue", "lightgreen", "pink"),
        vertex.cex = 2, 
        label.pos = 5, 
        vertex.border = F,
        edge.col = bme.col, 
        edge.lwd = 5, 
        diag = T,
        coord = coord.mat, 
        jitter = F,
        main = paste0("Block matrix image: Dice similarity ", stage[k])
  )
  
}

# 1: incumbents
# 2: governing parties (GPs)
# 3: ENGOs
# 4: residual group (RG)

# Differences in comparison to Jaccard coefficient:
# Stage 1: both 2-3 (3-2) and 2-4 (4-2) are statistically significantly (sig.) higher instead of not sig., i.e. 2 differences out of the 10 relationships
# Stage 2: 2-4 (4-2) not sig. (instead of sig. lower), i.e. 1 difference out of the 10 relationships
# Stage 3: 1-1 sig. higher instead of not sig., both 2-3 (3-2) and 2-4 (4-2) not sig. instead of sig. lower, i.e. 3 differences out of the 10 relationships
# In total: 5 differences out of the 20
# Relationships between incumbents and GPs as well as incumbents and ENGOs were stable

# ---- coalition detection ----

# ---- 1. stage ----

# (1) affiliation to the rescission options

# rescission options
indu11 <- which(bipnet1[,"limits rescission is beneficial"]>0)
indu12 <- which(bipnet1[,"option 2"]>0)
indu13 <- which(bipnet1[,"option 3"]>0)
indu14 <- which(bipnet1[,"option 4"]>0)
i11 <- union(indu11, indu12)
i12 <- union(indu13, indu14)

# opposition to the rescission 
envi1 <- which(bipnet1[,"option 1"]>0) 

# actors who unambiguously support the rescission (one of the three options) 
data.frame(indu1=setdiff(rownames(bipnet1)[union(i11, i12)], rownames(bipnet1)[envi1])) 
indu1.lab <- setdiff(rownames(bipnet1)[union(i11, i12)], rownames(bipnet1)[envi1]) 
indu1.i <- match(indu1.lab, rownames(bipnet1))

# actors who unambiguously oppose the rescission (all three options) 
data.frame(setdiff(rownames(bipnet1)[envi1], rownames(bipnet1)[union(i11, i12)]))
envi1.lab <- setdiff(rownames(bipnet1)[envi1], rownames(bipnet1)[union(i11, i12)])
envi1.i <- match(envi1.lab, rownames(bipnet1))

# (2) actor embeddedness 

# reading in organizations congruence (via concepts) network (average activity normalization)

cong1 <- as.matrix(read.xlsx("org_cong_avactivity_010115-170515_1.xlsx", 
                              rowNames = T))

# distribution of edge weights
hist(cong1[cong1>0], xlim = c(0, round(max(cong1)+1, 0)), ylim = c(0, 80),
     breaks = 10*round(max(cong1)+1, 0))
quantile(cong1[cong1>0])

# slicing and sensitivity analysis

# m-slice: w > Q2
cong1 <- ifelse(cong1 > quantile(cong1[cong1>0], probs=0.5), 1, 0)

# threshold values change to Q0 (0.017), Q1 (0.395), Q2 (0.727), Q3 (1.333) produce same coalitions
# only if the threshold increases to 90th percentile (2.286)
# environmental coalition is reduced by 2 actors; industry coalition remains same

# are all group members connected after the slicing?

isolates(cong1)
rownames(cong1)[isolates(cong1)]

which(indu1.i == isolates(cong1))
indu.coalition1 <- indu1.lab

which(envi1.i == isolates(cong1))
envi.coalition1 <- envi1.lab

# industry coalition (N = 18)
indu.coalition1

# environmental coalition (N = 14)
envi.coalition1

# block modeling: are the groups cohesive?

v1 <- rep(NA, nrow(bipnet1))
v1[indu1.i] <- 1
v1[envi1.i] <- 2
v1[is.na(v1)] <- 3

gden(cong1, mode = "graph")
bm <- blockmodel(cong1, ec = v1, block.content = "density", mode = "graph")
round(bm$block.model, 3)

# the within-blocks for coalitions show higher density than the overall (network) average up to w > Q3 (incl.)
# only for w > 90th industry coalition exhibits lower density than the overall (network) average

# ---- 2. stage ----

# (1) affiliation to the rescission options

# rescission options
indu21 <- which(bipnet2[,"limits rescission is beneficial"]>0)
indu22 <- which(bipnet2[,"option 2"]>0)
indu23 <- which(bipnet2[,"option 3"]>0)
indu24 <- which(bipnet2[,"option 4"]>0)
i21 <- union(indu21, indu22)
i22 <- union(indu23, indu24)

# opposition to the rescission 
envi2 <- which(bipnet2[,"option 1"]>0) 

# actors who unambiguously support the rescission (one of the three options) 
data.frame(indu2=setdiff(rownames(bipnet2)[union(i21, i22)], rownames(bipnet2)[envi2])) 
indu2.lab <- setdiff(rownames(bipnet2)[union(i21, i22)], rownames(bipnet2)[envi2])
indu2.i <- match(indu2.lab, rownames(bipnet2))

# actors who unambiguously oppose the rescission (all three options) 
data.frame(envi2=setdiff(rownames(bipnet2)[envi2], rownames(bipnet2)[union(i21, i22)]))
envi2.lab <- setdiff(rownames(bipnet2)[envi2], rownames(bipnet2)[union(i21, i22)])
envi2.i <- match(envi2.lab, rownames(bipnet2))
rownames(bipnet2)[envi2.i]

# (2) actor embeddedness 

# reading in organizations congruence (via concepts) network (average activity normalization)

cong2 <- as.matrix(read.xlsx("org_cong_avactivity_180515-300815_2.xlsx", 
                              rowNames = T))

# distribution of edge weights
hist(cong2[cong2>0], xlim = c(0, round(max(cong2)+1, 0)), ylim = c(0, 20),
     breaks = 10*round(max(cong2)+1, 0))
quantile(cong2[cong2>0])

# slicing and sensitivity analysis

# m-slice: w > Q2
cong2 <- ifelse(cong2 > quantile(cong2[cong2>0], probs=0.5), 1, 0)

# threshold values change to Q0 (0.056), Q1 (0.223), Q2 (0.439), Q3 (0.667) produce same coalitions
# only if the threshold increases to 90th percentile (1.241)
# industry coalition is reduced by 2 actors and environmental coalition is reduced by 1 actor

# are all group members connected after the slicing?

isolates(cong2)
rownames(cong2)[isolates(cong2)]

which(indu2.i == isolates(cong2))
indu.coalition2 <- indu2.lab[-which(indu2.i == isolates(cong2))]

# industry coalition (N = 5)
indu.coalition2

which(envi2.i == isolates(cong2))
envi.coalition2 <- envi2.lab

# environmental coalition (N = 5) 
envi.coalition2 

# block modeling: are the groups cohesive?

v2 <- rep(NA, nrow(bipnet2))
v2[indu2.i] <- 1
v2[envi2.i] <- 2
v2[is.na(v2)] <- 3

gden(cong2, mode = "graph")
bm2 <- blockmodel(cong2, ec = v2, block.content = "density", mode = "graph")
round(bm2$block.model, 3)

# the within-blocks for coalitions show higher density than the overall (network) average up to w > Q3 (incl.)
# only for w > 90th industry coalition exhibits lower density than the overall (network) average

# ---- 3. stage ----

# (1) affiliation to the rescission options

# rescission options
indu31 <- which(bipnet3[,"limits rescission is beneficial"]>0)
indu32 <- which(bipnet3[,"option 2"]>0)
indu33 <- which(bipnet3[,"option 3"]>0)
indu34 <- which(bipnet3[,"option 4"]>0)
i31 <- union(indu31, indu32)
i32 <- union(indu33, indu34)

# opposition to the rescission 
envi3 <- which(bipnet3[,"option 1"]>0) 

# actors who unambiguously support the rescission (one of the three options) 
data.frame(indu3=setdiff(rownames(bipnet3)[union(i31, i32)], rownames(bipnet3)[envi3])) 
indu3.lab <- setdiff(rownames(bipnet3)[union(i31, i32)], rownames(bipnet3)[envi3])
indu3.i <- match(indu3.lab, rownames(bipnet3))

# actors who unambiguously oppose the rescission (all three options) 
data.frame(envi3=setdiff(rownames(bipnet3)[envi3], rownames(bipnet3)[union(i31, i32)]))
envi3.lab <- setdiff(rownames(bipnet3)[envi3], rownames(bipnet3)[union(i31, i32)])
envi3.i <- match(envi3.lab, rownames(bipnet3))
rownames(bipnet3)[envi3.i]

# (2) actor embeddedness 

# reading in organizations congruence (via concepts) network (average activity normalization)

cong3 <- as.matrix(read.xlsx("org_cong_avactivity_310815-191015_3.xlsx", 
                             rowNames = T))

# distribution of edge weights
hist(cong3[cong3>0], xlim = c(0, round(max(cong3)+1, 0)), ylim = c(0, 40),
     breaks = 10*round(max(cong3)+1, 0))
quantile(cong3[cong3>0])

# slicing and sensitivity analysis

# m-slice: w > Q2
cong3 <- ifelse(cong3 > quantile(cong3[cong3>0], probs=0.5), 1, 0)

# threshold values change to Q0 (0.0290), Q1 (0.250), Q2 (0.5) produce same coalitions
# if threshold increases to Q3 (1) industry coalition is reduced by 1 actor 
# if the threshold increases to 90th percentile (1.558) both coalitions are reduced by 1 actor each

# are all group members connected after the slicing? 

isolates(cong3)
rownames(cong3)[isolates(cong3)]

which(indu3.i == isolates(cong3))
indu.coalition3 <- indu3.lab

# industry coalition (N = 12)
indu.coalition3

which(envi3.i == isolates(cong3))
envi.coalition3 <- envi3.lab

# environmental coalition (N = 6)
envi.coalition3

# block modeling: are the groups cohesive?

v3 <- rep(NA, nrow(bipnet3))
v3[indu3.i] <- 1
v3[envi3.i] <- 2
v3[is.na(v3)] <- 3

gden(cong3, mode = "graph")
bm3 <- blockmodel(cong3, ec = v3, block.content = "density", mode = "graph")
round(bm3$block.model, 3)

# the within-blocks for coalitions show higher density than the overall (network) average up to w > 90th (incl.)

# ---- discursive strategies ----

# reading in classification scheme: which concepts belong to specific strategies (defined in table 1)

strat <- read.xlsx("classification_scheme.xlsx", colNames = T)
strat

sec.agr <- strat$concept[which(strat$strategy == "securitization")]
sec.dis <- strat$concept[which(strat$strategy == "securitization (disagreement)")]
cap.agr <- strat$concept[which(strat$strategy == "capture")]
mas.agr <- strat$concept[which(strat$strategy == "masking")]
mas.dis <- strat$concept[which(strat$strategy == "masking (disagreement)")]
rei.agr <- strat$concept[which(strat$strategy == "reinvention")]

inc <- actors$org[which(actors$type=="incm")]

inc1.i <- match(inc, rownames(bipnet1))
p1 <- bipnet1[inc1.i,]
inc2.i <- as.integer(na.omit(match(inc, rownames(bipnet2))))
p2 <- bipnet2[inc2.i,]
inc3.i <- as.integer(na.omit(match(inc, rownames(bipnet3))))
p3 <- bipnet3[inc3.i,]

# ---- capture ----

# capture (agreements)
p1[,match(cap.agr, colnames(p1))]
p1.cap.agr <- sum(p1[,match(cap.agr, colnames(p1))], na.rm = T)
p1.cap.agr # 4 statements

p2[,match(cap.agr, colnames(p2))]
p2.cap.agr <- sum(p2[,match(cap.agr, colnames(p2))], na.rm = T)
p2.cap.agr # 0 statements

p3[,match(cap.agr, colnames(p3))]
p3.cap.agr <- sum(p3[,match(cap.agr, colnames(p3))], na.rm = T)
p3.cap.agr # 1 statement

# capture strategy share on the incumbents' total 
sum(p1.cap.agr, p2.cap.agr, p3.cap.agr) / sum(sum(abs(p1)), sum(abs(p2)), sum(abs(p3)))
# 3%

# stage 1
sum(p1.cap.agr) / sum(abs(p1)) # 4%
# stage 2
sum(p2.cap.agr) / sum(abs(p2)) # 0%
# stage 3
sum(p3.cap.agr) / sum(abs(p3)) # 2%

# ---- masking ----

# masking (agreements)
p1[,match(mas.agr, colnames(p1))]
p1.mas.agr <- sum(p1[,match(mas.agr, colnames(p1))], na.rm = T)
p1.mas.agr # 2 statements

p2[,match(mas.agr, colnames(p2))]
p2.mas.agr <- sum(p2[,match(mas.agr, colnames(p2))], na.rm = T)
p2.mas.agr # 1 statement

p3[,match(mas.agr, colnames(p3))]
p3.mas.agr <- sum(p3[,match(mas.agr, colnames(p3))], na.rm = T)
p3.mas.agr # 1 statement

# masking (disagreements)
p1[,match(mas.dis, colnames(p1))]
p1.mas.dis <- sum(abs(p1[,match(mas.dis, colnames(p1))]), na.rm = T)
p1.mas.dis # 1 statement

p2[,match(mas.dis, colnames(p2))]
p2.mas.dis <- sum(abs(p2[,match(mas.dis, colnames(p2))]), na.rm = T)
p2.mas.dis # 2 statements

p3[,match(mas.dis, colnames(p3))]
p3.mas.dis <- sum(abs(p3[,match(mas.dis, colnames(p3))]), na.rm = T)
p3.mas.dis # 0 statements

# masking strategy share on incumbents' total 
sum(p1.mas.agr, p2.mas.agr, p3.mas.agr, p1.mas.dis, p2.mas.dis, p3.mas.dis) / sum(sum(abs(p1)), sum(abs(p2)), sum(abs(p3)))
# 4%

# stage 1
sum(p1.mas.agr, p1.mas.dis) / sum(abs(p1)) # 3%
# stage 2
sum(p2.mas.agr, p1.mas.dis) / sum(abs(p2)) # 6%
# stage 3
sum(p3.mas.agr, p1.mas.dis) / sum(abs(p3)) # 4%

# masking strategy as the absence of agreements with negative impacts of coal mining and use
bipnet1.agr <- bipnet1
bipnet2.agr <- bipnet2
bipnet3.agr <- bipnet3
bipnet1.agr[bipnet1.agr<0] <- 1
bipnet2.agr[bipnet2.agr<0] <- 1
bipnet3.agr[bipnet3.agr<0] <- 1
sum(bipnet1.agr[, match(mas.dis, colnames(bipnet1))], na.rm = T)
sum(bipnet2.agr[, match(mas.dis, colnames(bipnet2))], na.rm = T)
sum(bipnet3.agr[, match(mas.dis, colnames(bipnet3))], na.rm = T)

rowSums(bipnet1.agr[inc1.i, match(mas.dis, colnames(bipnet1))])
# 1
rowSums(bipnet2.agr[inc2.i, na.omit(match(mas.dis, colnames(bipnet2)))])
# 2
rowSums(bipnet3.agr[inc3.i, na.omit(match(mas.dis, colnames(bipnet3)))])
# 0

sum(
  rowSums(bipnet1.agr[, match(mas.dis, colnames(bipnet1))]),
  rowSums(bipnet2.agr[, na.omit(match(mas.dis, colnames(bipnet2)))]),
  rowSums(bipnet3.agr[, na.omit(match(mas.dis, colnames(bipnet3)))])
)
# 73 agreement statements on negative impacts of coal mining and use

# ---- reinvention ----

# reinvention (agreements only)
p1[,match(rei.agr, colnames(p1))]
which(p1[,match(rei.agr, colnames(p1))] > 0)
p1.rei.agr <-sum(p1[which(p1[,match(rei.agr, colnames(p1))] > 0)]) 
p1.rei.agr # 21 statements

p2[,match(rei.agr, colnames(p2))]
which(p2[,match(rei.agr, colnames(p2))] > 0)
p2.rei.agr <-sum(p2[which(p2[,match(rei.agr, colnames(p2))] > 0)])
p2.rei.agr # 5 statements

p3[,match(rei.agr, colnames(p3))]
which(p3[,match(rei.agr, colnames(p3))] > 0)
p3.rei.agr <- sum(p3[which(p3[,match(rei.agr, colnames(p3))] > 0)])
p3.rei.agr # 8 statements

# reinvention strategy share on the incumbents' total 
sum(p1.rei.agr, p2.rei.agr, p3.rei.agr) / sum(sum(abs(p1)), sum(abs(p2)), sum(abs(p3)))
# 18%

# stage 1
sum(p1.rei.agr) / sum(abs(p1)) # 19%
# stage 2
sum(p2.rei.agr) / sum(abs(p2)) # 14%
# stage 3
sum(p3.rei.agr) / sum(abs(p3)) # 18%

# ---- securitization ----

# securitization (agreements)
p1[,match(sec.agr, colnames(p1))]
which(p1[,match(sec.agr, colnames(p1))] > 0)
p1.sec.agr <- sum(p1[which(p1[,match(sec.agr, colnames(p1))] > 0)])
p1.sec.agr # 30 statements

p2[,match(sec.agr, colnames(p2))]
which(p2[,match(sec.agr, colnames(p2))] > 0)
p2.sec.agr <- sum(p2[which(p2[,match(sec.agr, colnames(p2))] > 0)])
p2.sec.agr # 14 statements

p3[,match(sec.agr, colnames(p3))]
which(p3[,match(sec.agr, colnames(p3))] > 0)
p3.sec.agr <- sum(p3[which(p3[,match(sec.agr, colnames(p3))] > 0)])
p3.sec.agr # 15 statements

# securitization (disagreements)
p1[,match(sec.dis, colnames(p1))]
# concept not present at all

p2[,match(sec.dis, colnames(p2))]
# 0 statements

p3[,match(sec.dis, colnames(p3))]
# 0 statements

# securitization strategy share on the incumbents' total 
sum(p1.sec.agr, p2.sec.agr, p3.sec.agr) / sum(sum(abs(p1)), sum(abs(p2)), sum(abs(p3)))
# 31%

# stage 1
sum(p1.sec.agr) / sum(abs(p1)) # 27%
# stage 2
sum(p2.sec.agr) / sum(abs(p2)) # 39%
# stage 3
sum(p3.sec.agr) / sum(abs(p3)) # 33%

sum(colSums(abs(p1))) # 110 statements in stage 1 used by incumbents
sum(colSums(abs(p2))) # 36 statements in stage 2 used by incumbents
sum(colSums(abs(p3))) # 45 statements in stage 3 used by incumbents

# 95 codes (strategies)

p1.strat <- c(
  # capture agreements
  capture=sum(p1[,na.omit(match(cap.agr, colnames(p1)))]),   
  # masking agreements
  masking=sum(p1[,na.omit(match(mas.agr, colnames(p1)))]) + sum(abs(p1[,na.omit(match(mas.dis, colnames(p1)))])),                                 
  # reinvention agreements
  reinvention=sum(p1[,na.omit(match(rei.agr, colnames(p1)))]),      
  # securitization agreements + securitization disagreements
  securitization=sum(p1[,na.omit(match(sec.agr, colnames(p1)))]) + sum(abs(p1[,na.omit(match(sec.dis, colnames(p1)))]))  
)
p1.strat

p2.strat <- c(
  # capture agreements
  capture=sum(p2[,na.omit(match(cap.agr, colnames(p2)))]),   
  # masking agreements
  masking=sum(p2[,na.omit(match(mas.agr, colnames(p2)))]) + sum(abs(p2[,na.omit(match(mas.dis, colnames(p2)))])),                                 
  # reinvention agreements
  reinvention=sum(p2[,na.omit(match(rei.agr, colnames(p2)))]),      
  # securitization agreements + securitization disagreements
  securitization=sum(p2[,na.omit(match(sec.agr, colnames(p2)))]) + sum(abs(p2[,na.omit(match(sec.dis, colnames(p2)))]))  
)
p2.strat

p3.strat <- c(
  # capture agreements
  capture=sum(p3[,na.omit(match(cap.agr, colnames(p3)))]),   
  # masking agreements
  masking=sum(p3[,na.omit(match(mas.agr, colnames(p3)))]) + sum(abs(p3[,na.omit(match(mas.dis, colnames(p3)))])),                                 
  # reinvention agreements
  reinvention=sum(p3[,na.omit(match(rei.agr, colnames(p3)))]),      
  # securitization agreements + securitization disagreements
  securitization=sum(p3[,na.omit(match(sec.agr, colnames(p3)))]) + sum(abs(p3[,na.omit(match(sec.dis, colnames(p3)))]))  
)
p3.strat

bnet <- read.xlsx("bipnet_subtr.xlsx", rowNames = T)
colnames(bnet) <- gsub("\\.", " ", colnames(bnet))

total.strat <- c(
  capture=sum(bnet[inc,cap.agr]),                                      # capture agreements
  masking=sum(bnet[inc,mas.agr]) + sum(abs(bnet[inc,mas.dis])),        # masking agreements + masking disagreements
  reinvention=sum(bnet[inc,rei.agr]),                                  # reinvention agreements
  securitization=sum(bnet[inc,sec.agr]) + sum(abs(bnet[inc,sec.dis]))  # securitization agreements + securitization disagreements
)

strat <- unlist(c(p1.strat, p2.strat, p3.strat, total.strat))
strat <- strat[c(1,5,9,13,
                 2,6,10,14,
                 3,7,11,15,
                 4,8,12,16
)]

# Figure 5: Frequencies of incumbents' discursive strategies 
 
bp <- barplot(strat,
              col = c("lightgreen", "lightblue", "orangered", "grey",
                      "lightgreen", "lightblue", "orangered", "grey",
                      "lightgreen", "lightblue", "orangered", "grey"),
              width = c(0.05, 0.05, 0.05, 0.05, 0.05, 0.05, 0.05, 0.05,
                        0.05, 0.05, 0.05, 0.05, 0.05, 0.05, 0.05, 0.05),
              space = c(0.01, 0.01, 0.01, 0.01, 
                        1, 0.01, 0.01, 0.01,
                        1, 0.01, 0.01, 0.01,
                        1, 0.01, 0.01, 0.01), 
              xaxt='n', yaxt='n', ylim = c(0,70), cex.axis = 1.5
)

text(bp, strat+1.5, labels=c(strat), cex=1)
axis(side = 1, at = c(0.1,0.35,0.61,0.86), tick = F, cex.axis = 1,
     labels = c("capture", "masking", "reinvention", "securitization"))
legend(x=-0.05, y=65, inset=.05, c("First stage","Second stage", "Third stage", "Total"), 
       y.intersp = 0.8, x.intersp = 0.2, cex = 1,
       fill=c("lightgreen", "lightblue", "orangered", "grey"), border = F, bty = "n")

# ---- block model (Jaccard similarity) ----

at2 <- car::recode(actors$type2, 
                  recodes = "'incm' = '1'; 'dm' = '2'; 'engo' = '3'; 'pol' = '4'; 
                  'reg' = '5'; 'civ' = '6'; 'oth' = '7'")
at2
data.frame(actors$org, at2)

stage <- c("(Stage 1)", "(Stage 2)", "(Stage 3)")
group.names <- c("incumbents", "GPs", "ENGOs", "pol parties", "regional governance", "local actors", "remainder")
bipnet.list <- list(bipnet1, bipnet2, bipnet3)

for (k in 1:length(bipnet.list)) {
  
  # bipartite actorsXconcepts network for particular stage
  m <- as.matrix(bipnet.list[[k]]) 
  
  # dichotomization of the bipartite actorsXconcepts network
  m <- ifelse(m > 0, 1, 0)
  
  # matching present actors with the actor types
  at.i <- match(rownames(m), actors$org)
  # checking whether everything matches
  rownames(m)[which(is.na(match(rownames(m), actors$org))==T)]
  # saving the matches into the predefined actor groups
  groups <- at2[at.i] 
  
  # calculating the distance matrix based on Jaccard coefficient
  dm <- as.matrix(arules::dissimilarity(m, method = "Jaccard"))
  
  # getting proximity matrix for easier interpretation
  dm <- 1 - dm
  
  # block model
  
  # the proximity matrix is blocked based on the four predefined actor groups
  # the cell values are within- and between-block means of the Jaccard similarity coefficient 
  
  bm <- blockmodel(dm, ec = groups, mode = "graph")
  bm$block.model <- matrix(bm$block.model[!is.nan(bm$block.model)], byrow = T, nrow=length(table(groups)), ncol=length(table(groups)))
  bm
  
  # resulting block model
  round(bm$block.model, 3)
  
  # simulating a sampling distribution consisting of Jaccard similarity values obtained from 5000 random bipartite networks 
  # random networks have same number of nodes and are conditioned on density of the observed network
  
  jacc.dist <- c()
  set.seed(666)
  for (i in 1:5000) {
    
    # random bipartite graph
    rg <- igraph::sample_bipartite(n1 = nrow(m), n2 = ncol(m), p=sum(m)/(nrow(m)*ncol(m)))
    im <- igraph::as_incidence_matrix(rg)
    # Jaccard similarity coefficient matrix
    output <- as.matrix(arules::dissimilarity(im, method = "Jaccard"))
    jacc.mat <- 1 - output 
    jacc.dist[i] <- mean(jacc.mat) # overall (network) average
    
  }
  
  # constructing confidence intervals as 0.025 and 0.975 quantiles of the simulated sampling distribution
  par(mfrow=c(1,1))
  ci.l <- quantile(jacc.dist, probs = 0.025) # confidence interval lower bound
  ci.u <- quantile(jacc.dist, probs = 0.975) # confidence interval upper bound
  hist(jacc.dist, xlim = c(0,1))             # plotting of the simulated sampling distribution
  abline(v=ci.l, lwd=2)                      # confidence interval lower bound
  abline(v=ci.u, lwd=2)                      # confidence interval upper bound
  
  # visualization of image graph
  
  # saving image matrix
  bm.mat <- bm$block.model 
  
  # color coding of the image matrix based on the statistical in/significance (sig.) of the relationships
  bme.col <- matrix(rep("", nrow(bm.mat)*ncol(bm.mat)), nrow=nrow(bm.mat))
  bme.col[bm.mat>=ci.u] <- "lightgreen"               # sig. higher than the overall average
  bme.col[bm.mat<=ci.l] <- "red"                      # sig. lower than the overall average
  bme.col[bm.mat>ci.l & bm.mat<ci.u] <- "lightgrey"   # not sig. different than the overall average
  bme.col
  
  # plotting the block image graph
  gplot(matrix(rep(1, nrow(bm.mat)*nrow(bm.mat)), nrow(bm.mat), ncol(bm.mat)), 
        usearrows = F, 
        label = group.names[as.integer(names(table(groups)))], 
        label.cex = 2,
        vertex.col = "lightblue",
        vertex.cex = 2, 
        label.pos = 5, 
        vertex.border = F,
        edge.col = bme.col, 
        edge.lwd = 5, 
        diag = T,
        jitter = F,
        main = paste0("Block matrix image: Jaccard similarity ", stage[k])
  )
  
}

# To gain additional insights, we decomposed the residual group into four actor groups: 
# non-governing political parties (opposition and regional)
# regional governance (esp. regional government, municipalities)
# local civil actors 
# the remaining actors 

# Stage 1

# The results show that ENGOs position is sig. more similar to non-governing parties as well as to regional and local organizations. 
# Interestingly, there is a sig. higher similarity between the governing and non-governing (esp. regional) parties.
# It shows that the GPs, especially the junior coalition partners, also share some of the more critical stances on the limits rescission. 

# Stage 2

# Since the number of actors was markedly lower than in the previous period, the residual group could be divided only into opposition parties and remaining actors. 
# The between-group block of the ENGOs and non-governing parties again showed a sig. higher similarity. 

# Stage 3

# The RG was divided into three actor groups: pol parties (non-governing), regional governance, and remaining actors (a single research organization). 
# The between-group blocks with incumbents and GPs are sig. less similar. 
# The ENGOs are again aligned (sig. higher similarity) with the opposition parties and the research organization (Charles University Environment Centre). 

# ---- end of the script ----
